 function HarmoniqueSpherique(l,m,varargin)
%%  HarmoniqueSpherique
%
%   Ce programme calcule et trace le module, et son carr (densit angulaire de probabilit), 
%   des harmoniques sphriques d'expression gnrale :
%
%   Y(l,m)=((2l+1)(l-m)!/((l+m)!4pi))^0.5 P(l,m)exp(i*m*phi)
% 
%   Entres :   l = moment cintique orbital (Nombre quantique)
%               m = Nombre quantique magntique (|m|<l)
%
% Olivier Pujol : septembre 2013

%% Vrification

if (abs(m) > l),error('Mauvaise valeur de l ou m : il faut m <= l !');end

%% Dfinition du graphique tridimensionnel

% Nombre de points --> Rsolution angulaire : 2*pi/(NbPoints-1) = 0,01*pi
if (isempty(varargin)==1),NbPoints = 201;end     
if (length(varargin)==1),NbPoints = varargin{1};end     
if (length(varargin)>1),error('Mauvais nombre d''arguments optionnels');end     

PHI = linspace(0,2*pi,NbPoints);      % Angle azimutal (phi) des coordonnes sphriques
THETA = linspace(0,pi,NbPoints);    % Colatitude (theta) des coordonnes sphriques

[phi,theta] = meshgrid(PHI,THETA);  % Grille du graphique

clear PHI THETA

%% Calcul du polynme de Legendre correspondant

% La fonction "legendre(l,X)", interne  Matlab, calcule P(l,m) pour chaque 
% lment (k) de X, compris imprativement entre -1 et 1.
% La matrice obtenue est de dimension (l+1)*length(X) ;  
% chacun de ses lments P(m+1,k) correspond  P(l,m) valu en X(k). 
% Soulignons que pour la premire ligne de P, m=0.

P=legendre(l,cos(theta));          % Polynme de Legendre

%% Calcul du module de l'harmonique sphrique considre

Facteur =((2*l+1)*factorial(l-abs(m))/(4*pi*factorial(l+abs(m))))^0.5; % Rel positif

Ylm=nan(NbPoints,NbPoints);

if (l == 0)
    Ylm = Facteur.*P;  % Dans ce cas, P = 1 quel que soit phi (|Y(0,0)|)
else
    Plm=P(abs(m)+1,:);  % P(m+1,k) = P(l,m). Ici, Plm = matrice (NbPoints * NbPoints)
    iplm=1;             % Initialisation de l'indice de Plm 
    for iphi=1:NbPoints
        for itheta=1:NbPoints
            Ylm(itheta,iphi) = Facteur*Plm(iplm).*exp(j*abs(m)*phi(itheta,iphi));   % Y(l,m)
            iplm=iplm+1;
        end
    end
end

if (m<0)
    Ylm=(-1)^m*conj(Ylm);
end

%% Calcul des parties relle et imaginaire de l'harmonique sphrique

ReelY=real(Ylm);
ImagY=imag(Ylm);
ModY=abs(Ylm);

%% Calcul de la densit angulaire de probabilit

DaProba=ModY.^2;    % |Y(l,m)|^2

%% Reprsentations graphiques

% 1) Module |Y(l,m)|

    % Passage en coordonnes cartsiennes pour le trac
X=cos(phi).*sin(theta).*ModY;
Y=sin(phi).*sin(theta).*ModY;
Z=cos(theta).*ModY;

    % Dtermination des coordonnes maximales pour l'tendue des axes
xmax=max(max(X,[],1));
ymax=max(max(Y,[],1));
zmax=max(max(Z,[],1));

vmax = max([xmax ymax zmax]);   % Valeur maximale (valeur minimale = -vmax)

    % Figure
figure;
surf(X,Y,Z,ModY);
shading interp;
colormap(jet);
colorbar('vertical');
axis equal;
title(['|Y(l,m)| avec l  = ',num2str(l),' et m = ',num2str(m),''],'Fontsize',12);
xlabel('x','Fontsize',12);
ylabel('y','Fontsize',12);
zlabel('z','Fontsize',12);
axis([-vmax vmax -vmax vmax -vmax vmax]);

% Diagramme polaire dans un plan quelconque (ex : phi=0) contenant l'axe Oz
% On slectionne la premire colonne de la donne  tracer (ici le module de Y) 
% puis on utilise la fonction polar. Attention, l'angle polaire est
% pi/2-theta puisque theta est l'angle des coordonnes sphriques. 

figure
polar(pi/2-theta(:,1),ModY(:,1));
hold on
polar(pi/2-(pi+theta(:,1)),ModY(:,1));  % Partie symtrique en theta+pi
line([0 0],[-max(ModY(:,1)) max(ModY(:,1))],'Color','r','Linewidth',2)
title(['Polaire de |Y(l,m)| avec l = ',num2str(l),' et m = ',num2str(m),' dans un plan contenant l''axe Oz'],'Fontsize',12);

clear X Y Z xmax ymax zmax vmax

% 2) Densit angulaire de probabilit |Y(l,m)|^2

    % Passage en coordonnes cartsiennes pour le trac
X=cos(phi).*sin(theta).*DaProba;
Y=sin(phi).*sin(theta).*DaProba;
Z=cos(theta).*DaProba;

    % Dtermination des coordonnes maximales pour l'tendue des axes
xmax=max(max(X,[],1));
ymax=max(max(Y,[],1));
zmax=max(max(Z,[],1));

vmax = max([xmax ymax zmax]);   % Valeur maximale (valeur minimale = -vmax)

    % Figure
figure;
surf(X,Y,Z,DaProba)
shading interp;
colormap(jet);
colorbar('vertical');
axis equal;
title(['|Y(l,m)|^2 avec l = ',num2str(l),' et m = ',num2str(m),''],'Fontsize',12);
xlabel('x','Fontsize',12);
ylabel('y','Fontsize',12);
zlabel('z','Fontsize',12);
axis([-vmax vmax -vmax vmax -vmax vmax]);

% Diagramme polaire dans un plan quelconque (ex : phi=0) contenant l'axe Oz
% On slectionne la premire colonne de la donne  tracer (ici le module au carr de Y) 
% puis on utilise la fonction polar. Attention, l'angle polaire est
% pi/2-theta puisque theta est l'angle des coordonnes sphriques. 

figure
polar(pi/2-theta(:,1),DaProba(:,1));
hold on
polar(pi/2-(pi+theta(:,1)),DaProba(:,1));   % Partie symtrique en theta+pi
line([0 0],[-max(DaProba(:,1)) max(DaProba(:,1))],'Color','r','Linewidth',2)
title(['Polaire de Y(l,m)^2 avec l = ',num2str(l),' et m = ',num2str(m),' dans un plan contenant l''axe Oz'],'Fontsize',12)
clear X Y Z xmax ymax zmax vmax

% 3) Partie relle (on procde comme pour le module)

X=cos(phi).*sin(theta).*abs(ReelY);
Y=sin(phi).*sin(theta).*abs(ReelY);
Z=cos(theta).*abs(ReelY);

xmax=max(max(X,[],1));
ymax=max(max(Y,[],1));
zmax=max(max(Z,[],1));

vmax = max([xmax ymax zmax]);

figure
surf(X,Y,Z,ReelY);
shading interp;
colormap(jet);
colorbar('vertical');
axis equal;
title(['Partie relle de Y(l,m) avec l = ',num2str(l),' et m =  ',num2str(m),''],'Fontsize',12);
xlabel('x','Fontsize',12);
ylabel('y','Fontsize',12);
zlabel('z','Fontsize',12);
axis([-vmax vmax -vmax vmax -vmax vmax]);

clear X Y Z xmax ymax zmax vmax

if (m == 0)
    return;     % Si m=0, la partie imaginaire est nulle
end

% 4) Partie imaginaire (on procde comme pour le module)

X=cos(phi).*sin(theta).*abs(ImagY);
Y=sin(phi).*sin(theta).*abs(ImagY);
Z=cos(theta).*abs(ImagY);

xmax=max(max(X,[],1));
ymax=max(max(Y,[],1));
zmax=max(max(Z,[],1));

vmax = max([xmax ymax zmax]);

figure;
surf(X,Y,Z,ImagY);
shading interp;
colormap(jet);
colorbar('vertical');
axis equal;
title(['Partie imaginaire de Y(l,m) avec l = ',num2str(l),' et m = ',num2str(m),''],'Fontsize',12);
xlabel('x','Fontsize',12);
ylabel('y','Fontsize',12);
zlabel('z','Fontsize',12);
axis([-vmax vmax -vmax vmax -vmax vmax]);

clear X Y Z xmax ymax zmax vmax
